define("innovalog.code-editor", [], function () {

    var remoteLint = function (text, callback) {
        AJS.$.ajax({
            url: AJS.contextPath() + "/rest/" + addonShortKey + "/1/groovy/syntax",
            dataType: 'json',
            data: JSON.stringify({sourceCode: text}),
            type: 'POST',
            contentType: 'application/json',
            headers: {
                "Accept": "application/json"
            },
            success: function (response) {
                var annotations = [];
                if (response) {
                    response.forEach(function (error) {
                        annotations.push({
                            'message': error.message,
                            'severity': error.severity,
                            'from': CodeMirror.Pos(error.startLine - 1, error.startColumn - 1),
                            'to': CodeMirror.Pos(error.endLine - 1, error.endColumn - 1)
                        });
                    });
                }
                callback(annotations);
            },
            error: function (e) {
                callback([]);
            }
        });
    };
    remoteLint.async = true;
    CodeMirror.registerHelper("lint", "groovy", remoteLint);

    CodeMirror.defineMIME("groovy-template", {name: "htmlembedded", scriptingModeSpec:"groovy"});
    CodeMirror.defineMIME("jql-groovy-template", {name: "htmlembedded", scriptingModeSpec:"groovy"});

    //a folder for Groovy imports that supports missing semicolons
    CodeMirror.registerHelper("fold", "groovy-import", function (cm, start) {
        function hasImport(line) {
            if (line < cm.firstLine() || line > cm.lastLine()) return null;
            var start = cm.getTokenAt(CodeMirror.Pos(line, 1));
            if (!/\S/.test(start.string)) start = cm.getTokenAt(CodeMirror.Pos(line, start.end + 1));
            if (start.type != "keyword" || start.string != "import") return null;
            // Now find either closing semicolon or end of line that does not end in a dot, return its position
            for (var i = line, e = Math.min(cm.lastLine(), line + 10); i <= e; ++i) {
                var text = cm.getLine(i), semi = text.indexOf(";");
                if (semi != -1) return {startCh: start.end, end: CodeMirror.Pos(i, semi)};
                if (!text.trim().endsWith('.')) return {startCh: start.end, end: CodeMirror.Pos(i, text.length)};
            }
        }

        var startLine = start.line, has = hasImport(startLine), prev;
        if (!has || hasImport(startLine - 1) || ((prev = hasImport(startLine - 2)) && prev.end.line == startLine - 1))
            return null;
        for (var end = has.end; ;) {
            var next = hasImport(end.line + 1);
            if (next == null) break;
            end = next.end;
        }
        return {from: cm.clipPos(CodeMirror.Pos(startLine, has.startCh + 1)), to: end};
    });
    CodeMirror.extendMode('groovy', {fold: ["brace", "groovy-import"]});

    function CodeEditor() {
    }

    CodeEditor.prototype.testScript = function (issueKeys, currentValueDef, maxResults, allowChanges, callback) {
        AJS.$.ajax({
            url: AJS.contextPath() + "/rest/" + addonShortKey + "/1/groovy/test",
            dataType: 'json',
            data: JSON.stringify({
                sourceCode: this.codeMirrorInstance.getValue(),
                language: this.codeMirrorInstance.getOption("mode"),
                issueKeys: issueKeys,
                currentValue: currentValueDef,
                maxResults: maxResults,
                allowChanges: !!allowChanges
            }),
            type: 'POST',
            contentType: 'application/json',
            headers: {
                "Accept": "application/json"
            },
            success: function (response) {
                callback(response);
            },
            error: function (e) {
                if (e.status == 401)
                    callback({errorMessage: "Only Administrators can test scripts."});
                else
                    callback({errorMessage: "Script testing failed with a server error."});
            }
        });
    };

    CodeEditor.prototype.getParent = function (issueKey, callback) {
        AJS.$.ajax({
            url: AJS.contextPath() + "/rest/api/2/issue/" + issueKey + "?fields=parent",
            type: 'GET',
            contentType: 'application/json',
            headers: {
                "Accept": "application/json"
            },
            success: function (response) {
                callback(response.fields && response.fields.parent && response.fields.parent.key);
            },
            error: function (e) {
                callback(undefined);
            }
        });
    };

    CodeEditor.prototype.getLinkedIssues = function (issueKey, linkType, callback) {
        var linkTypeId = linkType.split(":")[1];
        var linkDirection = linkType.split(":")[0];
        AJS.$.ajax({
            url: AJS.contextPath() + "/rest/" + addonShortKey + "/1/groovy/linkedIssues/" + issueKey,
            type: 'GET',
            contentType: 'application/json',
            data: {
                linkDirection: linkDirection,
                linkTypeId: linkTypeId
            },
            headers: {
                "Accept": "application/json"
            },
            success: function (response) {
                callback(response);
            },
            error: function (e) {
                callback([]);
            }
        });
    };

    CodeEditor.prototype.wrapTextArea = function (textArea, editorMode) {
        if (!this.codeMirrorInstance) {
            var self = this;
            const $textArea = AJS.$(textArea);
            var cm = CodeMirror.fromTextArea($textArea.get(0), {
                lineNumbers: true,
                mode: editorMode || "groovy",
                viewportMargin: Infinity,
                foldGutter: true,
                gutters: ['CodeMirror-lint-markers', 'CodeMirror-linenumbers', 'CodeMirror-foldgutter'],
                autoCloseBrackets: true,
                lint: true
            });

            this.codeMirrorInstance = cm;

            if ($textArea.is("[required]") || $textArea.is("[data-aui-validation-required]")) {
                const $cm = AJS.$(cm.getWrapperElement());
                $cm.attr("id", "CodeMirror-editor-" + this.fieldName);
                $textArea.attr("data-aui-validation-displayfield", "CodeMirror-editor-" + this.fieldName);
            }

            const toolbarTpl = $textArea.prev();
            const toolbar = toolbarTpl.clone();
            toolbar.show();
            this.toolbar = cm.addPanel(toolbar.get(0));

            toolbar.find("#btn-find").click(function (e) {
                e.preventDefault();
                this.blur();
                cm.execCommand('find');
            });
            toolbar.find("#btn-find-next").click(function (e) {
                e.preventDefault();
                this.blur();
                cm.execCommand('findNext');
            });
            toolbar.find("#btn-replace").click(function (e) {
                e.preventDefault();
                this.blur();
                cm.execCommand('replace');
            });

            var testDialogSelector = "#test-dialog-" + this.fieldName;
            var dialog = AJS.dialog2(testDialogSelector);
            toolbar.find("#testScript").click(function (e) {
                this.blur();
                dialog.show();
                e.preventDefault();
            });
            toolbar.find("#test-again").click(function (e) {
                e.preventDefault();
                this.blur();
                self.doTest();
            });
        }
        this.setMode(editorMode);
    };

    CodeEditor.prototype.unwrapTextArea = function () {
        if (!this.codeMirrorInstance)
            return; //already unwrapped
        var textArea = AJS.$(this.codeMirrorInstance.getTextArea());

        this.toolbar.clear();
        delete this.toolbar;

        this.codeMirrorInstance.toTextArea();

        delete this.codeMirrorInstance;

        this.setMode('text');

        if (textArea.is("[required]") || textArea.is("[data-aui-validation-required]")) {
            textArea.removeAttr("data-aui-validation-displayfield");
        }

        const toolbar = textArea.prev();
        toolbar.find("#btn-find").off("click");
        toolbar.find("#btn-find-next").off("click");
        toolbar.find("#btn-replace").off("click");
    };

    CodeEditor.prototype.setMode = function (editorMode) {
        if (this.codeMirrorInstance) {
            this.codeMirrorInstance.setOption("mode", editorMode);

            if (editorMode == "groovy") {
                //fold import block
                var line = 0;
                var start = this.codeMirrorInstance.getTokenAt(CodeMirror.Pos(line, 1));
                if (!/\S/.test(start.string)) start = this.codeMirrorInstance.getTokenAt(CodeMirror.Pos(line, start.end + 1));
                if (start.type == 'comment') {
                    line = 1;
                    start = this.codeMirrorInstance.getTokenAt(CodeMirror.Pos(line, 1));
                    if (!/\S/.test(start.string)) start = this.codeMirrorInstance.getTokenAt(CodeMirror.Pos(line, start.end + 1));
                }
                if (start.type == "keyword" && start.string == "import") {
                    this.codeMirrorInstance.foldCode(CodeMirror.Pos(line, 0), null, "fold");
                }
            }
        }

        //adjust appearance
        if (editorMode == 'groovy') {
            this.codeMirrorInstance.setOption("lint", true);
            this.codeMirrorInstance.performLint();
            AJS.$(this.fieldGroupSelector + " .script-help").show();
            AJS.$(this.fieldGroupSelector + " .groovy-help").show();
            AJS.$(this.fieldGroupSelector + " .gsp-help").hide();
            AJS.$(this.fieldGroupSelector + " .text-help").hide();
            AJS.$(this.fieldGroupSelector + " .template-help").hide();
            AJS.$(this.fieldGroupSelector + " .editor-footer").show();
            AJS.$(this.fieldGroupSelector + " #testScript span#label").text('Test Groovy Script...');
            AJS.$(this.fieldGroupSelector + " .editor-needs-box").addClass('editor-box');
        } else if (editorMode == 'groovy-template') {
            this.codeMirrorInstance.setOption("lint", false);
            AJS.$(this.fieldGroupSelector + " .script-help").hide();
            AJS.$(this.fieldGroupSelector + " .groovy-help").hide();
            AJS.$(this.fieldGroupSelector + " .gsp-help").show();
            AJS.$(this.fieldGroupSelector + " .text-help").hide();
            AJS.$(this.fieldGroupSelector + " .template-help").show();
            AJS.$(this.fieldGroupSelector + " .editor-footer").show();
            AJS.$(this.fieldGroupSelector + " #testScript span#label").text('Test Groovy template...');
            AJS.$(this.fieldGroupSelector + " .editor-needs-box").addClass('editor-box');
        } else if (editorMode == 'jql-groovy-template') {
            this.codeMirrorInstance.setOption("lint", false);
            AJS.$(this.fieldGroupSelector + " .script-help").hide();
            AJS.$(this.fieldGroupSelector + " .groovy-help").hide();
            AJS.$(this.fieldGroupSelector + " .gsp-help").show();
            AJS.$(this.fieldGroupSelector + " .text-help").hide();
            AJS.$(this.fieldGroupSelector + " .template-help").show();
            AJS.$(this.fieldGroupSelector + " .editor-footer").show();
            AJS.$(this.fieldGroupSelector + " #testScript span#label").text('Test JQL search...');
            AJS.$(this.fieldGroupSelector + " .editor-needs-box").addClass('editor-box');
        } else {
            AJS.$(this.fieldGroupSelector + " .script-help").hide();
            AJS.$(this.fieldGroupSelector + " .text-help").show();
            AJS.$(this.fieldGroupSelector + " .template-help").hide();
            AJS.$(this.fieldGroupSelector + " .editor-footer").hide();
            AJS.$(this.fieldGroupSelector + " .editor-needs-box").removeClass('editor-box');
        }
    };

    CodeEditor.prototype.isWrapped = function () {
        return !!this.codeMirrorInstance;
    };

    CodeEditor.prototype.save = function () {
        if (this.codeMirrorInstance)
            this.codeMirrorInstance.save();
    };

    CodeEditor.prototype.init = function (fieldName, showParent, linkTypeSelector, currentValueFieldSelector, currentValueIssue, allowChanges) {
        var codeEditor = this;
        var testDialogSelector = "#test-dialog-" + fieldName;
        this.fieldGroupSelector = "#" + fieldName + "-group";
        this.fieldName = fieldName;
        var dialog = AJS.dialog2(testDialogSelector);

        AJS.$(codeEditor.fieldGroupSelector + " #" + fieldName).data('editor', this);

        AJS.$(testDialogSelector + " #dialog-close-button").click(function (e) {
            e.preventDefault();
            dialog.hide();
        });

        AJS.$(testDialogSelector + " #dialog-submit-button").click(function (e) {
            e.preventDefault();
            dialog.hide();
            doTest();
        });

        AJS.$(testDialogSelector + " form.aui").submit(function (evt) {
            evt.preventDefault();
            dialog.hide();
            doTest();
        });

        if (showParent)
            AJS.$(testDialogSelector + " #issue").on("change", function () {
                codeEditor.getParent(AJS.$(testDialogSelector + " #issue").val(), function (parentKey) {
                    AJS.$(testDialogSelector + " #parentIssue").val(parentKey || "");
                })
            });

        if (linkTypeSelector)
            if (linkTypeSelector=="%jql")
                {
                    const refreshLinkedIssueList = function () {
                        var issueKey = AJS.$(testDialogSelector + " #issue").val();
                        if (Array.isArray(issueKey))
                            issueKey = issueKey[0];
                        var jql = AJS.$("#jql").data("editor").codeMirrorInstance.getValue();

                        AJS.$.ajax({
                            url: AJS.contextPath() + "/rest/" + addonShortKey + "/1/groovy/test",
                            dataType: 'json',
                            data: JSON.stringify({
                                sourceCode: jql,
                                language: "jql-groovy-template",
                                issueKeys: {issue:issueKey},
                                maxResults: AJS.$("#maxIssues").val()
                            }),
                            type: 'POST',
                            contentType: 'application/json',
                            headers: {
                                "Accept": "application/json"
                            },
                            success: function (result) {
                                var selectList = AJS.$(testDialogSelector + " #linkedIssue");
                                selectList.html("");
                                if (result.issues)
                                    result.issues.forEach(function (linkedIssue) {
                                        selectList.append("<option value='" + linkedIssue.key + "'>" + linkedIssue.key + " - " + linkedIssue.summary + "</option>")
                                    });
                            }
                        });
                    };
                    AJS.$(testDialogSelector + " #issue, #maxIssues").on("change", refreshLinkedIssueList);
                    AJS.$("#jql").data("editor").codeMirrorInstance.on("blur", refreshLinkedIssueList);
                }
            else
                AJS.$(testDialogSelector + " #issue"+ (!linkTypeSelector.startsWith("*:") ? ", " + linkTypeSelector : "")).on("change", function () {
                    var issueKey = AJS.$(testDialogSelector + " #issue").val();
                    if (!issueKey)
                        AJS.$(testDialogSelector + " #linkedIssue").html("");
                    else {
                        const linkType = linkTypeSelector.startsWith("*:") ? linkTypeSelector : AJS.$(linkTypeSelector).val();
                        codeEditor.getLinkedIssues(issueKey, linkType, function (linkedIssues) {
                            var selectList = AJS.$(testDialogSelector + " #linkedIssue");
                            selectList.html("");
                            linkedIssues.forEach(function (linkedIssue) {
                                selectList.append("<option value='" + linkedIssue.key + "'>" + linkedIssue.key + " - " + linkedIssue.summary + "</option>")
                            });
                        })
                    }
                });

        //initialize our issue picker field
        var ip = new SingleIssuePicker({
            element: AJS.$(testDialogSelector + " #issue"),
            //userEnteredOptionsMsg: "toto",
            uppercaseUserEnteredOnSelect: true,
            submitInputVal: true
        });

        //move the test-dialog outside its current FORM, and wrap its content with an AUI FORM (because forms cannot be put inside forms)
        AJS.$(testDialogSelector).prependTo("body").wrapInner("<form class='aui' onsubmit='e.preventDefault();dialog.hide();doTest();'></form>");

        function doTest() {
            AJS.$("#" + fieldName + "-message").closeMessage();
            AJS.messages.info(codeEditor.fieldGroupSelector + " .message-location", {
                title: "Testing your script/template...",
                id: fieldName + "-message"
            });
            var issueKey = AJS.$(testDialogSelector + " #issue").val();
            if (Array.isArray(issueKey))
                issueKey = issueKey[0];
            var issueKeys = {
                issue: issueKey
            };
            if (showParent)
                issueKeys.parentIssue = AJS.$(testDialogSelector + " #parentIssue").val();
            if (linkTypeSelector)
                issueKeys.linkedIssue = AJS.$(testDialogSelector + " #linkedIssue").val();
            var currentValueDef;
            if (currentValueFieldSelector)
                currentValueDef = {
                    field: AJS.$(currentValueFieldSelector).val(),
                    sourceIssue: currentValueIssue
                };

            var language = codeEditor.codeMirrorInstance.getOption("mode");
            var maxResults;
            if (language=="jql-groovy-template")
                maxResults = AJS.$("#maxIssues").val();

            codeEditor.testScript(issueKeys, currentValueDef, maxResults, allowChanges, function (result) {
                AJS.$("#" + fieldName + "-message").closeMessage();
                AJS.$(codeEditor.fieldGroupSelector + " #test-again").show();

                if (result.errorMessage) {
                    var errorMsg = AJS.$('<div/>').text(result.errorMessage).html();
                    var body = "<h5>Message:</h5><div class='message-box'><pre>" + errorMsg + "</pre></div>";
                    if (result.stack) {
                        var stack = AJS.$('<div/>').text(result.stack).html();
                        body += "<h5>Stack:</h5><div class='code-box'><pre>" + stack + "</pre></div>";
                    }
                    if (language=="jql-groovy-template" && result.resultString) {
                        var resultStr = AJS.$('<div/>').text(result.resultString).html();
                        body += "<h5>Actual JQL search:</h5><div class='code-box'><pre>" + resultStr + "</pre></div>";
                    }
                    if (result.logs) {
                        var logs = AJS.$('<div/>').text(result.logs).html();
                        body += "<h5>Logs:</h5><div class='message-box'><pre>" + logs + "</pre></div>";
                    }
                    AJS.messages.error(codeEditor.fieldGroupSelector + " .message-location", {
                        title: "There was an error during the execution of your script",
                        body: body,
                        id: fieldName + "-message"
                    });
                    if (result.errorLine) {
                        codeEditor.codeMirrorInstance.addLineClass(result.errorLine - 1, "background", "error-line");
                        AJS.$(document).on("aui-message-close", function (e, a) {
                            if (a.attr("id") == fieldName + "-message")
                                codeEditor.codeMirrorInstance.removeLineClass(result.errorLine - 1, "background", "error-line");
                        });
                    }
                } else {
                    var resultStr = AJS.$('<div/>').text(result.resultString).html();
                    var body = "";
                    if (result.resultType)
                        body = "<h5>Result type:</h5><pre>" + result.resultType + "</pre>";
                    if (language=="jql-groovy-template")
                        body += "<h5>Actual JQL search:</h5><div class='code-box'><pre>" + resultStr + "</pre></div>";
                    else
                        body += "<h5>Result value:</h5><div class='code-box'><pre>" + resultStr + "</pre></div>";
                    if (result.issues) {
                        body += "<h5>Issues returned by your search:</h5><div class='code-box'><table class='plain-table'>";
                        result.issues.forEach(function (issue) {
                            body += "<tr><td>"+issue.key+"</td><td>"+issue.summary+"</td></tr>";
                        });
                        body += "</table></div>"
                    }
                    if (result.logs) {
                        var logs = AJS.$('<div/>').text(result.logs).html();
                        body += "<h5>Logs:</h5><div class='message-box'><pre>" + logs + "</pre></div>";
                    }
                    AJS.messages.success(codeEditor.fieldGroupSelector + " .message-location", {
                        title: language=="jql-groovy-template" ? "Your JQL search ran successfully" : "Your script ran successfully",
                        body: body,
                        id: fieldName + "-message"
                    });
                }
            })
        }

        this.doTest = doTest;
    };

    return function () {
        return new CodeEditor();
    }
});